#include "UserProfileSony.h"
#include "SignInSony.h"
#include "MessagePipe.h"
#include "ErrorCodesSony.h"


using namespace sce::Toolkit::NP;
using namespace sce::Toolkit::NP::Utilities;

namespace UnityPlugin
{
	CachedUserProfile gUserProfile;

	DO_EXPORT( ErrorCode, PrxRequestUserProfile ) ()
	{
		return gUserProfile.RequestUserProfile();
	}

	DO_EXPORT( bool, PrxUserProfileIsBusy ) ()
	{
		return gUserProfile.IsBusy();
	}

	DO_EXPORT( bool, PrxGetUserProfile ) (UserProfile* profile)
	{
		gUserProfile.GetUserProfile(profile);
		return true;
	}

	DO_EXPORT( ErrorCode, PrxRequestRemoteUserNpID ) (const char* onlineID)
	{
		return gUserProfile.RequestRemoteUserNpID(onlineID);
	}

	DO_EXPORT( bool, PrxGetRemoteUserNpID ) (NpID* npID)
	{
		return gUserProfile.GetRemoteUserNpID(npID);
	}

	DO_EXPORT( ErrorCode, PrxRequestRemoteUserProfileForOnlineID ) (const char* onlineID)
	{
		return gUserProfile.RequestRemoteUserProfileForOnlineID(onlineID);
	}

	DO_EXPORT( ErrorCode, PrxRequestRemoteUserProfileForNpID ) (const unsigned char* npID)
	{
		return gUserProfile.RequestRemoteUserProfileForNpID(npID);
	}

	DO_EXPORT( bool, PrxGetRemoteUserProfile ) (RemoteUserProfile* profile)
	{
		return gUserProfile.GetRemoteUserProfile(profile);
	}

	DO_EXPORT( bool, PrxUserProfileGetLastError ) (ResultCode* result)
	{
		return gUserProfile.GetLastError(result);
	}
	
	CachedUserProfile::CachedUserProfile()
		: m_GetCount(0)
		, m_State(STATE_NOTHING)
		, m_LastResult("UserProfile")
	{
	}

	bool CachedUserProfile::IsBusy()
	{
		SimpleLock::AutoLock lock(m_Lock);

		if(m_NpLookup.IsBusy())
		{
			return true;
		}
		return m_State != STATE_NOTHING;
	}

	void CachedUserProfile::GetUserProfile(UserProfile* profile)
	{
		SimpleLock::AutoLock lock(m_Lock);
		*profile = m_profile;
	}

	std::string CachedUserProfile::GetOnlineID()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_onlineID;
	}

	bool CachedUserProfile::ProcessEvent(const Event& event)
	{
		SimpleLock::AutoLock lock(m_Lock);
		bool handled = false;

		switch(event.event)
		{
		case Event::profileGotOnlineId:
			if(m_GetCount > 0)
			{
				switch(m_State)
				{
				case STATE_GETTING_LOCAL_PROFILE:
					m_GetCount --;
					if(m_FutureOnlineID.hasError())
					{
						m_LastResult.SetResultSCE(m_FutureOnlineID.getError(), true, __FUNCTION__, __LINE__);
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
						return true;
					}
					else
					{
						m_onlineID = m_FutureOnlineID.get()->data;
						m_profile.onlineID = m_onlineID.c_str();
					}
					break;

				default:
					m_LastResult.SetResult(NP_ERR_BAD_STATE, true, __FUNCTION__, __LINE__, "Unexpected State");
					break;
				}
			}

			handled = true;
			break;


		case Event::profileGotNpId:
			switch(m_State)
			{
				case STATE_GETTING_LOCAL_PROFILE:
					if(m_GetCount > 0)
					{
						m_GetCount --;
						if(m_FutureNpID.hasError())
						{
							m_LastResult.SetResultSCE(m_FutureNpID.getError(), true, __FUNCTION__, __LINE__);
							UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
							return true;
						}
						else
						{
							m_npID = *m_FutureNpID.get();
							m_profile.npID = (unsigned char*)&m_npID;
							m_profile.npIDSize = sizeof(SceNpId);
						}
					}
					break;


				default:
					m_LastResult.SetResult(NP_ERR_BAD_STATE, true, __FUNCTION__, __LINE__, "Unexpected State");
					break;
			}

			handled = true;
			break;

		case Event::profileGotAvatarUrl:
			if(m_GetCount > 0)
			{
				switch(m_State)
				{
				case STATE_GETTING_LOCAL_PROFILE:
					m_GetCount --;
					if(m_FutureAvatarURL.hasError())
					{
						m_LastResult.SetResultSCE(m_FutureAvatarURL.getError(), true, __FUNCTION__, __LINE__);
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
						return true;
					}
					else
					{
						m_avatarURL = m_FutureAvatarURL.get()->data;
						m_profile.avatarURL = m_avatarURL.c_str();
					}
					break;

				default:
					m_LastResult.SetResult(NP_ERR_BAD_STATE, true, __FUNCTION__, __LINE__, "Unexpected State");
					break;
				}
			}
			handled = true;
			break;

		case Event::profileGotCountryInfo:
			if(m_GetCount > 0)
			{
				switch(m_State)
				{
				case STATE_GETTING_LOCAL_PROFILE:
					m_GetCount --;
					if(m_FutureCountryInfo.hasError())
					{
						m_LastResult.SetResultSCE(m_FutureCountryInfo.getError(), true, __FUNCTION__, __LINE__);
						UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
						return true;
					}
					else
					{
						m_countryCode = m_FutureCountryInfo.get()->countryCode.data;
						m_profile.countryCode = m_countryCode.c_str();
						m_profile.language = m_FutureCountryInfo.get()->language;
					}
					break;

				default:
					m_LastResult.SetResult(NP_ERR_BAD_STATE, true, __FUNCTION__, __LINE__, "Unexpected State");
					break;
				}
			}
			handled = true;
			break;

		case Event::profileGotParentalInfo:
			if(m_GetCount > 0)
			{
				m_GetCount --;
				if(m_FutureParentInfo.hasError())
				{
					m_LastResult.SetResultSCE(m_FutureParentInfo.getError(), true, __FUNCTION__, __LINE__);
					UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
					return true;
				}
				else
				{
					m_profile.age = m_FutureParentInfo.get()->age;
					m_profile.chatRestricted = m_FutureParentInfo.get()->isChatRestricted;
					m_profile.contentRestricted = m_FutureParentInfo.get()->isContentRestricted;
				}
			}
			handled = true;
			break;

		default:
			break;

		}

		switch(m_State)
		{
		case STATE_GETTING_LOCAL_PROFILE:
			if(handled && m_GetCount == 0)
			{
				// Got everything so send the data ready message.
				UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_GotUserProfile);
				m_State = STATE_NOTHING;
			}
			break;

		}

		return handled;
	}

	ErrorCode CachedUserProfile::RequestUserProfile()
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		SimpleLock::AutoLock lock(m_Lock);
		int ret;

		m_LastResult.Reset();
		m_State = STATE_GETTING_LOCAL_PROFILE;

		m_FutureOnlineID.reset();
		m_FutureNpID.reset();
		m_FutureAvatarURL.reset();
		m_FutureCountryInfo.reset();
		m_FutureParentInfo.reset();

#define PROFILE_PARAMS true

		// Online ID...
		ret = sce::Toolkit::NP::UserProfile::Interface::getOnlineId( &m_FutureOnlineID, PROFILE_PARAMS );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_GetCount ++;

		// NpID...
		ret = sce::Toolkit::NP::UserProfile::Interface::getNpId( &m_FutureNpID, PROFILE_PARAMS );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_GetCount ++;

		// Avatar URL...
		ret = sce::Toolkit::NP::UserProfile::Interface::getAvatarUrl( &m_FutureAvatarURL, PROFILE_PARAMS );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_GetCount ++;

		// Country info...
		ret = sce::Toolkit::NP::UserProfile::Interface::getCountryInfo( &m_FutureCountryInfo, PROFILE_PARAMS );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_GetCount ++;

		// Parental Control info...
		ret = sce::Toolkit::NP::UserProfile::Interface::getParentalControlInfo( &m_FutureParentInfo, PROFILE_PARAMS );
		if (ret != SCE_TOOLKIT_NP_SUCCESS)
		{
			return m_LastResult.SetResultSCE(ret, true, __FUNCTION__, __LINE__);
		}
		m_GetCount ++;
		return m_LastResult.GetResult();
	}

	SceNpId CachedUserProfile::GetNpID()
	{
		SimpleLock::AutoLock lock(m_Lock);
		return m_npID;
	}

	void CachedUserProfile::Update()
	{
		NpLookup::NpLookupResult result = m_NpLookup.Update();
		switch(result)
		{
		case NpLookup::NPLOOKUP_OK:
		case NpLookup::NPLOOKUP_IDLE:
		case NpLookup::NPLOOKUP_BUSY:
			break;

		case NpLookup::NPLOOKUP_FAILED:
			m_LastResult.SetResultSCE(m_NpLookup.GetLastErrorCode(), true, m_NpLookup.GetLastErrorMessage());
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_UserProfileError);
			break;

		case NpLookup::NPLOOKUP_GOT_NPID:
			m_RemoteNpID = m_NpLookup.GetUserNpID();
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_GotRemoteUserNpID);
			break;

		case NpLookup::NPLOOKUP_GOT_USERINFO:
			m_RemoteNpID = m_NpLookup.GetUserNpID();
			m_RemoteUserInfo = m_NpLookup.GetUserInfo();
			m_RemoteAboutMe = m_NpLookup.GetAboutMe();
			m_RemoteLanguages = m_NpLookup.GetLanguages();
			m_RemoteCountryCode = m_NpLookup.GetCountryCode();
			UnityPlugin::Messages::AddMessage(UnityPlugin::Messages::kNPToolKit_GotRemoteUserProfile);
			break;

		default:
			m_LastResult.SetResult(NP_ERR_BAD_STATE, true, __FUNCTION__, __LINE__, "Unexpected state");
			break;
		}
	}


	ErrorCode CachedUserProfile::RequestRemoteUserNpID(const char* onlineID)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		SimpleLock::AutoLock lock(m_Lock);

		if(!gSignedInState.IsSignedIn())
		{
			return m_LastResult.SetResult(NP_ERR_NOT_SIGNED_IN, true);
		}

		m_LastResult.Reset();
		SceNpOnlineId userOnlineID;
		memset(&userOnlineID,0, sizeof(SceNpOnlineId));
		strncpy(userOnlineID.data, onlineID, SCE_NP_ONLINEID_MAX_LENGTH);
		NpLookup::NpLookupResult result = m_NpLookup.LookupUserNpID(m_npID, userOnlineID);
		if(result == NpLookup::NPLOOKUP_FAILED)
		{
			return m_LastResult.SetResultSCE(m_NpLookup.GetLastErrorCode(), true, m_NpLookup.GetLastErrorMessage());
		}
		return m_LastResult.GetResult();
	}
	
	bool CachedUserProfile::GetRemoteUserNpID(NpID* npID)
	{
		SimpleLock::AutoLock lock(m_Lock);
		npID->npID = (unsigned char*)&m_RemoteNpID;
		npID->npIDSize = sizeof(SceNpId);
		return true;
	}

	ErrorCode CachedUserProfile::RequestRemoteUserProfileForOnlineID(const char* onlineID)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		SimpleLock::AutoLock lock(m_Lock);

		if(!gSignedInState.IsSignedIn())
		{
			return m_LastResult.SetResult(NP_ERR_NOT_SIGNED_IN, true);
		}

		m_LastResult.Reset();
		SceNpOnlineId userOnlineID;
		memcpy(&userOnlineID, onlineID, sizeof(SceNpOnlineId));
		NpLookup::NpLookupResult result = m_NpLookup.LookupUserInfo(m_npID, userOnlineID);
		if(result == NpLookup::NPLOOKUP_FAILED)
		{
			return m_LastResult.SetResultSCE(m_NpLookup.GetLastErrorCode(), true, m_NpLookup.GetLastErrorMessage());
		}

		return m_LastResult.GetResult();
	}

	ErrorCode CachedUserProfile::RequestRemoteUserProfileForNpID(const unsigned char* npID)
	{
		if(IsBusy())
		{
			return m_LastResult.SetResult(NP_ERR_BUSY, true);
		}

		SimpleLock::AutoLock lock(m_Lock);

		if(!gSignedInState.IsSignedIn())
		{
			return m_LastResult.SetResult(NP_ERR_NOT_SIGNED_IN, true);
		}

		m_LastResult.Reset();
		SceNpId userNpID;
		memcpy(&userNpID, npID, sizeof(SceNpId));
		NpLookup::NpLookupResult result = m_NpLookup.LookupUserInfo(m_npID, userNpID);
		if(result == NpLookup::NPLOOKUP_FAILED)
		{
			return m_LastResult.SetResultSCE(m_NpLookup.GetLastErrorCode(), true, m_NpLookup.GetLastErrorMessage());
		}

		return m_LastResult.GetResult();
	}

	bool CachedUserProfile::GetRemoteUserProfile(RemoteUserProfile* profile)
	{
		profile->language = m_RemoteLanguages.language1;
		profile->onlineID = (const char*)&m_RemoteUserInfo.userId;		// This might be the onlineID, or it might be the npID.
		profile->npID = (const unsigned char*)&m_RemoteNpID;
		profile->npIDSize = sizeof(SceNpId);
		profile->avatarURL = (const char*)m_RemoteUserInfo.icon.data;
		profile->countryCode = (const char*)m_RemoteCountryCode.data;

		profile->firstName = (const char *)NULL;
		profile->middleName = (const char *)NULL;
		profile->lastName = (const char *)NULL;
		profile->profilePictureUrl = (const char *)NULL;

		return true;
	}


};
